/*
 * Shannon Duvall
 * PhotoGUI.java
 * This object keeps up with the original image,
 * the buttons, and their actions.
 */

package photomanipulation;
import processing.core.*;
import processing.xml.*;
import java.util.*;
import yourwork.*;

import java.awt.*;

public class PhotoGUI {
	
	public final int BUTTON_HEIGHT = 60;
	public final int BUTTON_WIDTH = 80;
	private final int ORIGINAL_INDEX = 0;
	private final int NEWPIC_INDEX = 1;
	
	// The button labels need to be IN THE SAME ORDER as the actions.  I connect
	// them by comparing their indices.  So myButtonNames[i] has action myActions[i].
	// Leave the "Original" button's action null.
	
	private String[] myButtonNames = {"Original", "New Pic", 
			"Color Swap","Red Filter", 
			"Negative", "Darker", 
			"Lighter", "Grayscale",
			"Posterize", "Tie Dye", 
			"Static"};
	private PhotoAction[] myActions = {null, null,
			new ColorSwap(), new RedFilter(), 
			new Negative(), new Darker(), 
			new Lighter(), new Grayscale(), 
			new Posterize(), new TieDye(), 
			new Static()};
	private Button[] myButtons;
	// Here are all the pictures so I can scroll between them.
	private ArrayList<PImage> myPictures;
	// Which picture am I on?
	private int myIndex;
	
	public PhotoGUI(ArrayList<PImage> pictures)
	{		
		myPictures = pictures;
		myIndex = 0;
		myButtons = new Button[myButtonNames.length];
		Point dimension = new Point(BUTTON_WIDTH, BUTTON_HEIGHT);
		for(int i = 0; i< myButtonNames.length; i++){
			Point upperLeft = new Point(i*BUTTON_WIDTH,0);
			myButtons[i] = new Button(myButtonNames[i],upperLeft,dimension,myActions[i]);
		}
	}
	
	// Draw the buttons.  The image will be drawn by the PApplet.
	public void draw(PApplet canvas){
		for(Button b:myButtons){
			b.draw(canvas);
		}		
	}
	
	// When the mouse is clicked, compute the new image and return it.
	public PImage mouseClicked(int mousex, int mousey, PImage current, PApplet canvas){
		int index = getButton(mousex,mousey);
		// If you didn't click on a button, don't change the image.
		if(index == -1){
			return current;
		}
		// Special cases for reverting to the original or getting a new picture
		if(index == ORIGINAL_INDEX){
			return myPictures.get(myIndex);
		}
		if(index == NEWPIC_INDEX){
			myIndex = (myIndex+1) % myPictures.size();
			return myPictures.get(myIndex);
		}
		// If the button hasn't been implemented yet, do nothing.
		if(myActions[index].computeColor(Color.WHITE)==null){
			return current;
		}
		
		// Make a new image
		PImage result = canvas.createImage(current.width,current.height,canvas.RGB);
		// Load current and new pixels.
		current.loadPixels();
		result.loadPixels();
		// For each pixel
		for (int x = 0; x < current.width; x++) {
		    for (int y = 0; y < current.height; y++ ) {
		      int loc = x + y*current.width;
		      // Do a lot of work to convert between java Colors and Processing colors... grr.
		      int red = (int)(canvas.red(current.pixels[loc]));
		      int green = (int)(canvas.green(current.pixels[loc]));
		      int blue = (int)(canvas.blue(current.pixels[loc]));
		      Color startColor = new Color(red,green,blue);
		      // Ask the action to convert the color
		      Color endColor = myActions[index].computeColor(startColor);
		      result.pixels[loc] = canvas.color(endColor.getRed(),endColor.getGreen(),endColor.getBlue());
		    }
		  }

    	// We changed the pixels in the result image
		result.updatePixels();
		return result;
	}
	
	// Calculate which button contains the point (x,y).  If no button is 
	// drawn there, return -1.
	public int getButton(int x, int y){
		if(x<0 || x >BUTTON_WIDTH*myButtons.length || y<0 || y>BUTTON_HEIGHT){
			return -1;
		}
		else{
			return x/BUTTON_WIDTH;
		}
	}
}
